import { useState } from 'react';
import { Badge, Button, Container } from 'react-bootstrap';
import { MdKeyboardArrowRight } from 'react-icons/md';
import { BsArrowRightShort } from 'react-icons/bs';
import _ from 'lodash';
import { getColor, getFiberInfo } from '../../../../../util/colorModule';
import FiberTractView from './FiberTractView';

function Component({ component, arrowLeft }) {
    switch (component?.type) {
        case 'splitter':
            return (
                <>
                    {arrowLeft && (
                        <MdKeyboardArrowRight className='text-white' />
                    )}
                    <Badge bg='danger'>
                        {_.capitalize(component.type)}:- {component.name} (1:
                        {component.split_ratio})
                    </Badge>
                    {!arrowLeft && (
                        <MdKeyboardArrowRight className='text-white' />
                    )}
                </>
            );
        case 'splitter_out':
            return (
                <>
                    {arrowLeft && (
                        <MdKeyboardArrowRight className='text-white' />
                    )}
                    <Badge bg='danger'>
                        {_.capitalize(component.splitter.type)}:-{' '}
                        {component.splitter.name} ({component.out_number}:{' '}
                        {getColor(component.out_number)})
                    </Badge>
                    {!arrowLeft && (
                        <MdKeyboardArrowRight className='text-white' />
                    )}
                </>
            );
        case 'port':
            return (
                <>
                    {' '}
                    {arrowLeft && (
                        <MdKeyboardArrowRight className='text-white' />
                    )}
                    <Badge bg='danger'>
                        {_.capitalize(component.device.type)}:-{' '}
                        {component.device.name} ({component.port_number}:{' '}
                        {component.label}
                        {component.port_stream === 'incoming' ? '⬇️' : '⬆️'})
                    </Badge>
                    {!arrowLeft && (
                        <MdKeyboardArrowRight className='text-white' />
                    )}
                </>
            );
        case 'onu':
            return (
                <>
                    {' '}
                    {arrowLeft && (
                        <MdKeyboardArrowRight className='text-white' />
                    )}
                    <Badge bg='danger'>
                        {_.capitalize(component.type)}:- {component.id}.{' '}
                        {component.name} ({component.mac_address})
                    </Badge>
                    {!arrowLeft && (
                        <MdKeyboardArrowRight className='text-white' />
                    )}
                </>
            );
        default:
            return null;
    }
}

const SingleFiberView = ({ fiber, cable, handleFiberDisconnect }) => {
    const [components, setComponents] = useState({
        startNode: null,
        endNode: null,
    });
    const {
        number,
        color: colorName,
        bgColor,
    } = getFiberInfo(fiber.fiber_number);
    const color = _.capitalize(colorName);
    return (
        <div
            className='p-2 shadow mt-1 rounded w-100 d-flex align-items-center'
            style={{ background: bgColor }}
        >
            <span className='p-2 text-light rounded me-2 bg-dark shadow'>
                {number}: {color}
            </span>
            <div className='bg-dark p-2 rounded'>
                <Component
                    component={fiber.parent_component || components?.startNode}
                />
                <FiberTractView
                    fiber={fiber.parent_fiber}
                    parent
                    setComponents={setComponents}
                />
                {fiber.parent_fiber && (
                    <MdKeyboardArrowRight className='text-white' />
                )}
                <Badge bg='primary border'>
                    {cable.name}: ({number}: {color})
                </Badge>{' '}
                {fiber.child_fiber && (
                    <MdKeyboardArrowRight className='text-white' />
                )}
                <FiberTractView
                    fiber={fiber.child_fiber}
                    setComponents={setComponents}
                />
                <Component
                    component={fiber.child_component || components?.endNode}
                    arrowLeft
                />
            </div>
            <Button
                variant='dark'
                className='ms-auto'
                onClick={() => handleFiberDisconnect(fiber.id)}
            >
                Disconnect
            </Button>
        </div>
    );
};

const FiberDetailView = ({ cableDetail, handleFiberDisconnect }) => {
    return (
        <Container fluid>
            {cableDetail.tubes.map((tube) => {
                return (
                    <div className='shadow p-2 rounded' key={tube.id}>
                        <h6>
                            Tube <BsArrowRightShort /> {tube.tube_number} - has{' '}
                            {tube.fiber_count} fibers
                        </h6>
                        <hr className='w-50 m-0' />
                        {tube.fibers.map((fiber) => {
                            return (
                                <SingleFiberView
                                    fiber={fiber}
                                    cable={cableDetail}
                                    key={fiber.id}
                                    handleFiberDisconnect={
                                        handleFiberDisconnect
                                    }
                                />
                            );
                        })}
                    </div>
                );
            })}
        </Container>
    );
};

export default FiberDetailView;
